'use strict';
function asyncGeneratorStep(gen, resolve, reject, _next, _throw, key, arg) {
    try {
        var info = gen[key](arg);
        var value = info.value;
    } catch (error) {
        reject(error);
        return;
    }
    if (info.done) {
        resolve(value);
    } else {
        Promise.resolve(value).then(_next, _throw);
    }
}
function _async_to_generator(fn) {
    return function() {
        var self = this, args = arguments;
        return new Promise(function(resolve, reject) {
            var gen = fn.apply(self, args);
            function _next(value) {
                asyncGeneratorStep(gen, resolve, reject, _next, _throw, "next", value);
            }
            function _throw(err) {
                asyncGeneratorStep(gen, resolve, reject, _next, _throw, "throw", err);
            }
            _next(undefined);
        });
    };
}
import MetatraderAccount from './metatraderAccount';
import MetatraderAccountReplica from './metatraderAccountReplica';
import { MetatraderAccountList } from './metatraderAccountApi';
let MetatraderAccountApi = class MetatraderAccountApi {
    /**
   * Returns trading accounts belonging to the current user, provides pagination in infinite scroll style
   * @param {AccountsFilter} [accountsFilter] optional filter
   * @return {Promise<Array<MetatraderAccount>>} promise resolving with MetaTrader account entities
   */ getAccountsWithInfiniteScrollPagination(accountsFilter) {
        var _this = this;
        return _async_to_generator(function*() {
            let accounts = yield _this._metatraderAccountClient.getAccounts(accountsFilter, '1');
            return accounts.map((a)=>new MetatraderAccount(a, _this._metatraderAccountClient, _this._metaApiWebsocketClient, _this._connectionRegistry, _this._expertAdvisorClient, _this._historicalMarketDataClient, _this._application));
        })();
    }
    /**
   * Returns trading accounts belonging to the current user with accounts count,
   * provides pagination in a classic style
   * @param {AccountsFilter} [accountsFilter] optional filter
   * @return {Promise<MetatraderAccountList>} promise resolving with an array of MetaTrader account entities and count
   */ getAccountsWithClassicPagination(accountsFilter) {
        var _this = this;
        return _async_to_generator(function*() {
            let accounts = yield _this._metatraderAccountClient.getAccounts(accountsFilter, '2');
            return {
                count: accounts.count,
                items: accounts.items.map((a)=>new MetatraderAccount(a, _this._metatraderAccountClient, _this._metaApiWebsocketClient, _this._connectionRegistry, _this._expertAdvisorClient, _this._historicalMarketDataClient, _this._application))
            };
        })();
    }
    /**
   * Returns trading account by id
   * @param {string} accountId MetaTrader account id
   * @return {Promise<MetatraderAccount>} promise resolving with MetaTrader account entity
   */ getAccount(accountId) {
        var _this = this;
        return _async_to_generator(function*() {
            let account = yield _this._metatraderAccountClient.getAccount(accountId);
            return new MetatraderAccount(account, _this._metatraderAccountClient, _this._metaApiWebsocketClient, _this._connectionRegistry, _this._expertAdvisorClient, _this._historicalMarketDataClient, _this._application);
        })();
    }
    /**
   * Returns trading account replica by trading account id and replica id
   * @param {string} accountId MetaTrader primary account id
   * @param {string} replicaId MetaTrader account replica id
   * @return {Promise<MetatraderAccountReplica>} promise resolving with MetaTrader account replica found
   */ getAccountReplica(accountId, replicaId) {
        var _this = this;
        return _async_to_generator(function*() {
            let account = yield _this._metatraderAccountClient.getAccount(accountId);
            let replica = yield _this._metatraderAccountClient.getAccountReplica(accountId, replicaId);
            return new MetatraderAccountReplica(replica, account, _this._metatraderAccountClient);
        })();
    }
    /**
   * Returns replicas for a trading account
   * @param {string} accountId Primary account id
   * @return {Promise<Array<MetatraderAccountReplica>>} promise resolving with MetaTrader account replicas found
   */ getAccountReplicas(accountId) {
        var _this = this;
        return _async_to_generator(function*() {
            let account = yield _this._metatraderAccountClient.getAccount(accountId);
            let replicas = yield _this._metatraderAccountClient.getAccountReplicas(accountId);
            if (replicas.items) {
                replicas = replicas.items;
            }
            return replicas.map((replica)=>new MetatraderAccountReplica(replica, account, _this._metatraderAccountClient));
        })();
    }
    /**
   * Adds a trading account and starts a cloud API server for the trading account
   * @param {NewMetatraderAccountDto} account MetaTrader account data
   * @return {Promise<MetatraderAccount>} promise resolving with created MetaTrader account entity
   */ createAccount(account) {
        var _this = this;
        return _async_to_generator(function*() {
            let id = yield _this._metatraderAccountClient.createAccount(account);
            return _this.getAccount(id.id);
        })();
    }
    /**
   * Constructs a MetaTrader account API instance
   * @param {MetatraderAccountClient} metatraderAccountClient MetaTrader account REST API client
   * @param {MetaApiWebsocketClient} metaApiWebsocketClient MetaApi websocket client
   * @param {ConnectionRegistry} connectionRegistry metatrader account connection registry
   * @param {ExpertAdvisorClient} expertAdvisorClient expert advisor REST API client
   * @param {HistoricalMarketDataClient} historicalMarketDataClient historical market data HTTP API client
   * @param {string} application application name
   */ constructor(metatraderAccountClient, metaApiWebsocketClient, connectionRegistry, expertAdvisorClient, historicalMarketDataClient, application){
        this._metatraderAccountClient = metatraderAccountClient;
        this._metaApiWebsocketClient = metaApiWebsocketClient;
        this._connectionRegistry = connectionRegistry;
        this._expertAdvisorClient = expertAdvisorClient;
        this._historicalMarketDataClient = historicalMarketDataClient;
        this._application = application;
    }
};
/**
 * Exposes MetaTrader account API logic to the consumers
 */ export { MetatraderAccountApi as default };

//# sourceMappingURL=data:application/json;base64,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