"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "default", {
    enumerable: true,
    get: function() {
        return ProvisioningProfileClient;
    }
});
const _fs = /*#__PURE__*/ _interop_require_default(require("fs"));
const _metaApiclient = /*#__PURE__*/ _interop_require_default(require("../metaApi.client"));
const _formdata = /*#__PURE__*/ _interop_require_default(require("form-data"));
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
let ProvisioningProfileClient = class ProvisioningProfileClient extends _metaApiclient.default {
    /**
   * Provisioning profile model
   * @typedef {Object} ProvisioningProfileDto
   * @property {String} _id provisioning profile unique identifier
   * @property {String} name provisioning profile name
   * @property {Number} version MetaTrader version (allowed values are 4 and 5)
   * @property {String} status provisioning profile status (allowed values are new and active)
   * @property {String} brokerTimezone broker timezone name from Time Zone Database
   * @property {String} brokerDSTSwitchTimezone broker DST switch timezone name from Time Zone Database
   */ /**
   * MT version
   * @typedef {4 | 5} Version
   */ /**
   * Provisioning profile type
   * @typedef {'mtTerminal' | 'managerApi'} ProvisioningProfileType
   */ /**
   * Provisioning profile status
   * @typedef {'new' | 'active'} ProvisioningProfileStatus
   */ /**
   * Get provisioning profiles API version
   * @typedef {'1' | '2'} GetProvisioningProfilesApiVersion
   */ /**
   * @typedef {Object} ProvisioningProfilesFilter
   * @property {number} [offset] Search offset (defaults to 0) (must be greater or equal to 0)
   * @property {number} [limit] Search limit (defaults to 1000) 
   * (must be greater or equal to 1 and less or equal to 1000)
   * @property {Version} [version] MT version
   * @property {ProvisioningProfileType} [type] Profile type
   * @property {ProvisioningProfileStatus} [status] Profile status
   * @property {string} [query] Partially search over provisioning profile name to match query
   */ /**
   * Provisioning profiles list model
   * @typedef {Object} ProvisioningProfilesListDto
   * @property {String} count provisioning profiles count
   * @property {Array<ProvisioningProfileDto>} items provisioning profiles list
   */ /**
   * Retrieves provisioning profiles owned by user
   * (see https://metaapi.cloud/docs/provisioning/api/provisioningProfile/readProvisioningProfiles/)
   * Method is accessible only with API access token
   * @param {ProvisioningProfilesFilter} [profilesFilter] provisioning profiles filter
   * @param {GetProvisioningProfilesApiVersion} [apiVersion] api version to use
   * @return {Promise<Array<ProvisioningProfileDto> | ProvisioningProfilesListDto>} promise resolving with provisioning profiles found
   */ getProvisioningProfiles(profilesFilter = {}, apiVersion) {
        if (this._isNotJwtToken()) {
            return this._handleNoAccessError("getProvisioningProfiles");
        }
        const opts = {
            url: `${this._host}/users/current/provisioning-profiles`,
            method: "GET",
            params: profilesFilter,
            headers: {
                "auth-token": this._token
            },
            json: true
        };
        if (apiVersion) {
            opts.headers["api-version"] = apiVersion;
        }
        return this._httpClient.request(opts, "getProvisioningProfiles");
    }
    /**
   * Retrieves a provisioning profile by id (see
   * https://metaapi.cloud/docs/provisioning/api/provisioningProfile/readProvisioningProfile/). Throws an error if
   * profile is not found.
   * Method is accessible only with API access token
   * @param {String} id provisioning profile id
   * @return {Promise<ProvisioningProfileDto>} promise resolving with provisioning profile found
   */ getProvisioningProfile(id) {
        if (this._isNotJwtToken()) {
            return this._handleNoAccessError("getProvisioningProfile");
        }
        const opts = {
            url: `${this._host}/users/current/provisioning-profiles/${id}`,
            method: "GET",
            headers: {
                "auth-token": this._token
            },
            json: true
        };
        return this._httpClient.request(opts, "getProvisioningProfile");
    }
    /**
   * New provisioning profile model
   * @typedef {Object} NewProvisioningProfileDto
   * @property {String} name provisioning profile name
   * @property {Number} version MetaTrader version (allowed values are 4 and 5)
   * @property {String} brokerTimezone broker timezone name from Time Zone Database
   * @property {String} brokerDSTSwitchTimezone broker DST switch timezone name from Time Zone Database
   */ /**
   * Provisioning profile id model
   * @typedef {Object} ProvisioningProfileIdDto
   * @property {String} id provisioning profile unique identifier
   */ /**
   * Creates a new provisioning profile (see
   * https://metaapi.cloud/docs/provisioning/api/provisioningProfile/createNewProvisioningProfile/). After creating a
   * provisioning profile you are required to upload extra files in order to activate the profile for further use.
   * Method is accessible only with API access token
   * @param {NewProvisioningProfileDto} provisioningProfile provisioning profile to create
   * @return {Promise<ProvisioningProfileIdDto>} promise resolving with an id of the provisioning profile created
   */ createProvisioningProfile(provisioningProfile) {
        if (this._isNotJwtToken()) {
            return this._handleNoAccessError("createProvisioningProfile");
        }
        const opts = {
            url: `${this._host}/users/current/provisioning-profiles`,
            method: "POST",
            headers: {
                "auth-token": this._token
            },
            json: true,
            data: provisioningProfile
        };
        return this._httpClient.request(opts, "createProvisioningProfile");
    }
    /**
   * Uploads a file to a provisioning profile (see
   * https://metaapi.cloud/docs/provisioning/api/provisioningProfile/uploadFilesToProvisioningProfile/). Uploading a
   * file by name is allowed only for Node.js.
   * Method is accessible only with API access token
   * @param {String} provisioningProfileId provisioning profile id to upload file to
   * @param {String} fileName name of the file to upload. Allowed values are servers.dat for MT5 profile, broker.srv for
   * MT4 profile
   * @param {String|Buffer} file path to a file to upload or buffer containing file contents
   * @return {Promise} promise resolving when file upload is completed
   */ uploadProvisioningProfileFile(provisioningProfileId, fileName, file) {
        if (this._isNotJwtToken()) {
            return this._handleNoAccessError("uploadProvisioningProfileFile");
        }
        const formData = new _formdata.default();
        if (typeof file === "string") {
            file = _fs.default.createReadStream(file);
            formData.append("file", file);
        } else {
            formData.append("file", file, "serverFile");
        }
        const opts = {
            method: "PUT",
            url: `${this._host}/users/current/provisioning-profiles/${provisioningProfileId}/${fileName}`,
            data: formData,
            headers: {
                ...formData.getHeaders(),
                "auth-token": this._token
            }
        };
        return this._httpClient.request(opts, "uploadProvisioningProfileFile");
    }
    /**
   * Deletes a provisioning profile (see
   * https://metaapi.cloud/docs/provisioning/api/provisioningProfile/deleteProvisioningProfile/). Please note that in
   * order to delete a provisioning profile you need to delete MT accounts connected to it first.
   * Method is accessible only with API access token
   * @param {String} id provisioning profile id
   * @return {Promise} promise resolving when provisioning profile is deleted
   */ deleteProvisioningProfile(id) {
        if (this._isNotJwtToken()) {
            return this._handleNoAccessError("deleteProvisioningProfile");
        }
        const opts = {
            url: `${this._host}/users/current/provisioning-profiles/${id}`,
            method: "DELETE",
            headers: {
                "auth-token": this._token
            },
            json: true
        };
        return this._httpClient.request(opts, "deleteProvisioningProfile");
    }
    /**
   * Updated provisioning profile data
   * @typedef {Object} ProvisioningProfileUpdateDto
   * @property {String} name provisioning profile name
   */ /**
   * Updates existing provisioning profile data (see
   * https://metaapi.cloud/docs/provisioning/api/provisioningProfile/updateProvisioningProfile/).
   * Method is accessible only with API access token
   * @param {String} id provisioning profile id
   * @param {ProvisioningProfileUpdateDto} provisioningProfile updated provisioning profile
   * @return {Promise} promise resolving when provisioning profile is updated
   */ updateProvisioningProfile(id, provisioningProfile) {
        if (this._isNotJwtToken()) {
            return this._handleNoAccessError("updateProvisioningProfile");
        }
        const opts = {
            url: `${this._host}/users/current/provisioning-profiles/${id}`,
            method: "PUT",
            headers: {
                "auth-token": this._token
            },
            json: true,
            data: provisioningProfile
        };
        return this._httpClient.request(opts, "updateProvisioningProfile");
    }
};

//# sourceMappingURL=data:application/json;base64,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